using UnityEngine;
using System;
using System.Runtime.InteropServices;
using System.Text;

namespace Sony
{
	namespace NP
	{
        public class Trophies
		{
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxTrophyGetLastError(out ResultCode result);
			
			public static bool GetLastError(out ResultCode result)
			{
				PrxTrophyGetLastError(out result);
				return result.lastError == ErrorCode.NP_OK;
			}

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxTrophyRegisterPackIsBusy();
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxTrophyRefreshGroupInfoIsBusy();
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxTrophyRefreshTrophyInfoIsBusy();
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxTrophyRefreshProgressIsBusy();

			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTrophyGetGameInfo(out GameInfo info);

			[DllImport("UnityNpToolkit")]
			private static extern void PrxTrophyLockList();
			[DllImport("UnityNpToolkit")]
			private static extern void PrxTrophyUnlockList();

			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTrophyRefreshGroupInfo();
			[DllImport("UnityNpToolkit")]
			private static extern int PrxTrophyGetGroupDetailsCount();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTrophyGetGroupDetails(int index, out GroupDetails details);
			[DllImport("UnityNpToolkit")]
			private static extern int PrxTrophyGetGroupDataCount();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTrophyGetGroupData(int index, out GroupData data);

			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTrophyRefreshTrophyInfo();
			[DllImport("UnityNpToolkit")]
			private static extern int PrxTrophyGetTrophyDetailsCount();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTrophyGetTrophyDetails(int index, out TrophyDetails details);
			[DllImport("UnityNpToolkit")]
			private static extern int PrxTrophyGetTrophyDataCount();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTrophyGetTrophyData(int index, out TrophyData data);

			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTrophyRefreshProgress();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTrophyGetProgress(out TrophyProgress info);

			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTrophyAward(int index);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxRegisterTrophyPack();

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct GameInfo
			{
				public int numGroups;
				public int numTrophies;
				public int numPlatinum;
				public int numGold;
				public int numSilver;
				public int numBronze;
				IntPtr _title;
				IntPtr _description;
				public string title { get { return Marshal.PtrToStringAnsi(_title); } }
				public string description { get { return Marshal.PtrToStringAnsi(_description); } }
			};

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct GroupDetails
			{
				public int groupId;
				public int numTrophies;
				public int numPlatinum;
				public int numGold;
				public int numSilver;
				public int numBronze;
				public int iconWidth;
				public int iconHeight;
				public int iconDataSize;
				IntPtr _iconData;
				IntPtr _title;
				IntPtr _description;
				public string title { get { return Marshal.PtrToStringAnsi(_title); } }
				public string description { get { return Marshal.PtrToStringAnsi(_description); } }
				public Texture2D icon
				{
					get
					{
						if (iconDataSize > 0)
						{
							byte[] bytes = new byte[iconDataSize];
							Marshal.Copy(_iconData, bytes, 0, iconDataSize);
							Texture2D icon = new Texture2D(iconWidth, iconHeight);
							icon.LoadImage(bytes);
							return icon;
						}
						return null;
					}
				}
				public bool hasIcon
				{
					get
					{
						return iconDataSize > 0;
					}
				}
			};

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct GroupData
			{
				public int groupId;
				public int unlockedTrophies;
				public int unlockedPlatinum;
				public int unlockedGold;
				public int unlockedSilver;
				public int unlockedBronze;
				public int progressPercentage;
			};

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct TrophyDetails
			{
				public int trophyId;
				public int trophyGrade;
				public int groupId;
				public bool hidden;
				IntPtr _name;
				IntPtr _description;
				public string name { get { return Marshal.PtrToStringAnsi(_name); } }
				public string description { get { return Marshal.PtrToStringAnsi(_description); } }
			};

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct TrophyData
			{
				public int trophyId;
				public bool unlocked;
				public long timestamp;
				public int iconWidth;
				public int iconHeight;
				public int iconDataSize;
				IntPtr _iconData;
				public Texture2D icon
				{
					get
					{
						if (iconDataSize > 0)
						{
							byte[] bytes = new byte[iconDataSize];
							Marshal.Copy(_iconData, bytes, 0, iconDataSize);
							Texture2D icon = new Texture2D(iconWidth, iconHeight);
							icon.LoadImage(bytes);
							return icon;
						}
						return null;
					}
				}
				public bool hasIcon
				{
					get
					{
						return iconDataSize > 0;
					}
				}
			};

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct TrophyProgress
			{
				public int unlockedTrophies;
				public int unlockedPlatinum;
				public int unlockedGold;
				public int unlockedSilver;
				public int unlockedBronze;
				public int progressPercentage;
			};

			static bool trophiesAreAvailable = false;

			public static event Messages.EventHandler OnPackageRegistered;
			public static event Messages.EventHandler OnRegisterPackageFailed;
			public static event Messages.EventHandler OnGotGameInfo;
			public static event Messages.EventHandler OnGotGroupInfo;
			public static event Messages.EventHandler OnGotTrophyInfo;
			public static event Messages.EventHandler OnGotProgress;
			public static event Messages.EventHandler OnUnlockedPlatinum;
			public static event Messages.EventHandler OnAwardedTrophy;
			public static event Messages.EventHandler OnAwardTrophyFailed;
			public static event Messages.EventHandler OnAlreadyAwardedTrophy;
			public static event Messages.EventHandler OnTrophyError;

			public static bool TrophiesAreAvailable
			{
				get { return trophiesAreAvailable; }
			}

			public static bool RegisterTrophyPackIsBusy()
			{
				return PrxTrophyRegisterPackIsBusy();
			}

			public static GameInfo GetCachedGameInfo()
			{
				GameInfo info = new GameInfo();
				PrxTrophyGetGameInfo(out info);
				return info;
			}

			public static bool RequestGroupInfoIsBusy()
			{
				return PrxTrophyRefreshGroupInfoIsBusy();
			}

			public static ErrorCode RequestGroupInfo()
			{
				return PrxTrophyRefreshGroupInfo();
			}

			public static GroupData[] GetCachedGroupData()
			{
				PrxTrophyLockList();
				int count = PrxTrophyGetGroupDataCount();
				GroupData[] trophies = new GroupData[count];
				for (int i = 0; i < count; i++)
				{
					PrxTrophyGetGroupData(i, out trophies[i]);
				}
				PrxTrophyUnlockList();

				return trophies;
			}

			public static GroupDetails[] GetCachedGroupDetails()
			{
				PrxTrophyLockList();
				int count = PrxTrophyGetGroupDetailsCount();
				GroupDetails[] trophies = new GroupDetails[count];
				for (int i = 0; i < count; i++)
				{
					PrxTrophyGetGroupDetails(i, out trophies[i]);
				}
				PrxTrophyUnlockList();

				return trophies;
			}

			public static bool RequestTrophyInfoIsBusy()
			{
				return PrxTrophyRefreshTrophyInfoIsBusy();
			}

			public static ErrorCode RequestTrophyInfo()
			{
				return PrxTrophyRefreshTrophyInfo();
			}

			public static TrophyData[] GetCachedTrophyData()
			{
				PrxTrophyLockList();
				int count = PrxTrophyGetTrophyDataCount();
				TrophyData[] trophies = new TrophyData[count];
				for (int i = 0; i < count; i++)
				{
					PrxTrophyGetTrophyData(i, out trophies[i]);
				}
				PrxTrophyUnlockList();

				return trophies;
			}

			public static TrophyDetails[] GetCachedTrophyDetails()
			{
				PrxTrophyLockList();
				int count = PrxTrophyGetTrophyDetailsCount();
				TrophyDetails[] trophies = new TrophyDetails[count];
				for (int i = 0; i < count; i++)
				{
					PrxTrophyGetTrophyDetails(i, out trophies[i]);
				}
				PrxTrophyUnlockList();

				return trophies;
			}

			public static bool RequestTrophyProgressIsBusy()
			{
				return PrxTrophyRefreshProgressIsBusy();
			}

			public static ErrorCode RequestTrophyProgress()
			{
				return PrxTrophyRefreshProgress();
			}

			public static TrophyProgress GetCachedTrophyProgress()
			{
				TrophyProgress progress = new TrophyProgress();
				PrxTrophyGetProgress(out progress);
				return progress;
			}

			public static ErrorCode AwardTrophy(int index)
			{
				return PrxTrophyAward(index);
			}

			public static ErrorCode RegisterTrophyPack()
			{
				return PrxRegisterTrophyPack();
			}

			public static bool ProcessMessage(Messages.PluginMessage msg)
			{
				switch (msg.type)
				{
					case Messages.MessageType.kNPToolKit_TrophySetSetupSuccess:
						trophiesAreAvailable = true;
						if (OnPackageRegistered != null) OnPackageRegistered(msg);
						return true;

					case Messages.MessageType.kNPToolKit_TrophySetSetupCancelled:
					case Messages.MessageType.kNPToolKit_TrophySetSetupAborted:
					case Messages.MessageType.kNPToolKit_TrophySetSetupFail:
						if (OnRegisterPackageFailed != null) OnRegisterPackageFailed(msg);
						break;

					case Messages.MessageType.kNPToolKit_TrophyGotGameInfo:
						if (OnGotGameInfo != null) OnGotGameInfo(msg);
						return true;

					case Messages.MessageType.kNPToolKit_TrophyGotGroupInfo:
						if (OnGotGroupInfo != null) OnGotGroupInfo(msg);
						return true;

					case Messages.MessageType.kNPToolKit_TrophyGotTrophyInfo:
						if (OnGotTrophyInfo != null) OnGotTrophyInfo(msg);
						return true;

					case Messages.MessageType.kNPToolKit_TrophyGotProgress:
						if (OnGotProgress != null) OnGotProgress(msg);
						return true;

					case Messages.MessageType.kNPToolKit_TrophyUnlocked:
						if (OnAwardedTrophy != null) OnAwardedTrophy(msg);
						return true;

					case Messages.MessageType.kNPToolKit_TrophyUnlockFailed:
						if (OnAwardTrophyFailed != null) OnAwardTrophyFailed(msg);
						return true;

					case Messages.MessageType.kNPToolKit_TrophyUnlockedAlready:
						if (OnAlreadyAwardedTrophy != null) OnAlreadyAwardedTrophy(msg);
						return true;

					case Messages.MessageType.kNPToolKit_TrophyUnlockedPlatinum:
						if (OnUnlockedPlatinum != null) OnUnlockedPlatinum(msg);
						return true;

					case Messages.MessageType.kNPToolKit_TrophyError:
						if(OnTrophyError != null) OnTrophyError(msg);
						return true;
				}

				return false;
			}
		}
	} // NP
} // Sony
